// Coverer.cpp : Defines the entry point for the console application.
//

// LDraw line coverer
// Cover two paths (ordered sequence of LDraw type 2 lines with triangle or quads).
// Optional lines are set-up between triangles and quads.
// Usage: coverer [-r] [-l <length>] [-b <bias>] LdrawLineFile1 LdrawLineFile2 LdrawSurfaceFileOut
// -r: Reverse order of second file.
// -l: Split segment lines longer than <length>
// -b: Bias triangle repartition towards one end or the other according to <bias [-100..100]>
// If input files are not ordered they will be ordered first.
// primitives must be inlined before usage if needed.

// Philippe "Philo" Hurbain - 2007 - www.philohome.com

// Contains excerpts of triangle intersection code from Tomas Mller
// See thesis here: http://www.cs.lth.se/home/Tomas_Akenine_Moller/pubs/tritri.pdf
// Complete code here: http://jgt.akpeters.com/papers/Moller97/tritri.html


// Contains softSurfer Line - Triangle intersection (adaptation)
// Copyright 2001, softSurfer (www.softsurfer.com)
// This code may be freely used and modified for any purpose
// providing that this copyright notice is included with it.
// SoftSurfer makes no warranty for this code, and cannot be held
// liable for any real or imagined damage resulting from its use.
// Users of this code must verify correctness for their application.

#include "stdafx.h"

/* Triangle/triangle intersection test routine,
 * by Tomas Moller, 1997.
 * See article "A Fast Triangle-Triangle Intersection Test",
 * Journal of Graphics Tools, 2(2), 1997
 * updated: 2001-06-20 (added line of intersection)
 *
 * This version computes the line of intersection as well (if they are not coplanar):
 * int tri_tri_intersect_with_isectline(double V0[3],double V1[3],double V2[3], 
 *				        double U0[3],double U1[3],double U2[3],int *coplanar,
 *				        double isectpt1[3],double isectpt2[3]);
 * coplanar returns whether the tris are coplanar
 * isectpt1, isectpt2 are the endpoints of the line of intersection
 */

#include <math.h>
#include <string.h>


#define USE_EPSILON_TEST TRUE  
#define EPSILON 0.000001


/* Null vector */

double null[3] = {0,0,0};

/* some macros */

#define CROSS(dest,v1,v2)                      \
              dest[0]=v1[1]*v2[2]-v1[2]*v2[1]; \
              dest[1]=v1[2]*v2[0]-v1[0]*v2[2]; \
              dest[2]=v1[0]*v2[1]-v1[1]*v2[0];

#define DOT(v1,v2) (v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2])

#define SUB(dest,v1,v2) dest[0]=v1[0]-v2[0]; dest[1]=v1[1]-v2[1]; dest[2]=v1[2]-v2[2]; 

#define ADD(dest,v1,v2) dest[0]=v1[0]+v2[0]; dest[1]=v1[1]+v2[1]; dest[2]=v1[2]+v2[2]; 

#define MULT(dest,v,factor) dest[0]=factor*v[0]; dest[1]=factor*v[1]; dest[2]=factor*v[2];

#define SET(dest,src) dest[0]=src[0]; dest[1]=src[1]; dest[2]=src[2]; 

#define MANHATTAN(v1, v2) (fabs(v1[0]-v2[0]) + fabs(v1[1]-v2[1]) + fabs(v1[2]-v2[2]))

#define DIST(v1, v2) (sqrt((v1[0]-v2[0])*(v1[0]-v2[0]) + (v1[1]-v2[1])*(v1[1]-v2[1]) + (v1[2]-v2[2])*(v1[2]-v2[2])))


// Tri_Angle computes the sine of the angle between the planes of two triangles.
// They are assumed to be non-degenerated
double Tri_Angle(double U0[3],double U1[3],double U2[3], double V0[3],double V1[3],double V2[3])
{
	double Unorm[3], Vnorm[3];
	double Temp[3];
	double U10[3], U20[3];
	double V10[3], V20[3];
	double len;
	SUB(U10, U1, U0);
	SUB(U20, U2, U0);
	SUB(V10, V1, V0);
	SUB(V20, V2, V0);
	CROSS(Temp, U10, U20);
	len = DIST(Temp, null);
	MULT(Unorm, Temp, 1/len);
	CROSS(Temp, V10, V20);
	len = DIST(Temp, null);
	MULT(Vnorm, Temp, 1/len);
	CROSS(Temp, Unorm, Vnorm);
	return(DIST(Temp, null));
}

// Check if a quad is convex and don't have aligned vertexes.
int Convex(double U0[3],double U1[3],double U2[3],double U3[3])
{
	double U01[3], U12[3], U23[3], U30[3];
	double T0[3], T1[3], T2[3], T3[3];
	int cnt=0;
	SUB(U01, U1, U0);
	SUB(U12, U2, U1);
	SUB(U23, U3, U2);
	SUB(U30, U0, U3);

	CROSS(T0, U30, U01);
	CROSS(T1, U01, U12);
	CROSS(T2, U12, U23);
	CROSS(T3, U23, U30);

	// Colinear Vertices
	if((DOT(T0, T0)<EPSILON) ||(DOT(T1, T1)<EPSILON) ||(DOT(T2, T2)<EPSILON) ||(DOT(T3, T3)<EPSILON)) return false;
	
	if(DOT(T0, T1) > 0) cnt++; else cnt--;
	if(DOT(T1, T2) > 0) cnt++; else cnt--;
	if(DOT(T2, T3) > 0) cnt++; else cnt--;
	if(DOT(T3, T0) > 0) cnt++; else cnt--;

	if(abs(cnt) == 4) return true;
	else return false;
}

#define MAX_TRI 1000
#define MAX_LINE 2000
#define SMALL 0.1
// previous coverer versions used 0.005 = sin(0.28)
// from version 1.3, uses 0.016 = sin(0.916), closer to specification limit.
#define SMALLANGLE 0.016

void usage(void)
{
	printf ("Usage: coverer [-r] [-s] [-b <bias>] [-l <length>] LdrawLine1 LdrawLine2 LdrawSurfaceOut\n");
	printf ("Options\n");
	printf ("  -s: Use sequential sweep (old method)\n");
	printf ("  -r: Reverse order of second file\n");
	printf ("  -l: Split segment lines longer than <length>\n");
	printf ("  -b: Bias triangle spreading towards one end\n"); 
	printf ("      or the other according to <bias [-100..100]>\n");
	printf("Press <Enter> to quit");
	getchar();
}

int main(int argc, char* argv[])
{

	double SortBuf[MAX_LINE][2][3];
	int next [MAX_LINE][2][2];
	double InLine1[MAX_LINE][2][3];
	double InLine2[MAX_LINE][2][3];
	double Surf[MAX_TRI][4][3];
	double CondLine[MAX_TRI][4][3];
	int CondFlag[MAX_TRI];
	int TriQuad[MAX_TRI];

	FILE *LdrawLineFile1;
	FILE *LdrawLineFile2;
	FILE *LdrawSurfaceFileOut;

	int NumTri, NumLine1, NumLine2, Degen, TriWritten;
	int InLineIdx;
	int NumPath;
	int NumCond, NumQuad;
	double d, dmax;

	char buf[1024];
	int type, color;

	int i, j, k, l;
	
	int flag=0; 
	int direction = false;
	int sort=false;
	int sequential=false;
	
	int circular;
	double maxlength = 100000;
	double bias=0;

	int line = 1;
	printf("\nLdraw Coverer v1.3 - by Philo");
	printf("\n-----------------------------\n");

	if(argc < 4) 		
	{
		usage();
		return 1;
	}
	for(i=1; i<argc-3; i++)
	{
		if(argv[i][0] != '-') 
		{
			usage();
			return 1;
		}
		switch (argv[i][1])
		{
		case 'r':
			direction = true;
			break;
		case 's':
			sequential = true;
			break;
		case 'l':
			i++;
			sscanf(argv[i], "%lf", &maxlength);
			if (maxlength<=0)
			{
				printf("Length (-l %.12lg) must be > 0\n", maxlength);
				printf("Press <Enter> to quit");
				getchar();
				return 3;
			}
			printf("l = %.12lg\n", maxlength);
			break;
		case 'b':
			i++;
			sscanf(argv[i], "%lf", &bias);
			if ((bias < -100) || (bias > 100))
			{
				printf("Bias (-b %.12lg) must be in [-100..100] range\n", bias);
				printf("Press <Enter> to quit");
				getchar();
				return 3;
			}
			printf("Bias = %.12lg\n", bias);
			break;
		default:
			usage();
			return 1;
		}
	}
	

	LdrawLineFile1 = fopen (argv[argc-3], "rt");
	if(LdrawLineFile1 == NULL)
	{
		printf("Can't open file\n%s\n", argv[argc-3]);
		printf("Press <Enter> to quit");
		getchar();
		return 2;
	}

	LdrawLineFile2 = fopen (argv[argc-2], "rt");
	if(LdrawLineFile2 == NULL)
	{
		printf("Can't open file\n%s\n", argv[argc-2]);
		printf("Press <Enter> to quit");
		getchar();
		return 2;
	}

	LdrawSurfaceFileOut = fopen (argv[argc-1], "wt");
	if(LdrawSurfaceFileOut == NULL)
	{
		printf("Can't create file\n%s\n", argv[argc-1]);
		printf("Press <Enter> to quit");
		getchar();
		return 2;
	}
	
	printf("\nRead Line File 1...\n");
	NumLine1=0;
	while (fgets(buf, 1024, LdrawLineFile1))
	{
		type = -1;
		sscanf(buf, "%d", &type);
		if(type==2)
		{
			sscanf (buf, "%d %d %lf %lf %lf %lf %lf %lf", &type, &color, 
				&SortBuf[NumLine1][0][0], &SortBuf[NumLine1][0][1], &SortBuf[NumLine1][0][2], 
				&SortBuf[NumLine1][1][0], &SortBuf[NumLine1][1][1], &SortBuf[NumLine1][1][2]);
				next[NumLine1][0][0] = next[NumLine1][1][0] = -1;
			NumLine1++;
		}
		if(NumLine1>MAX_LINE)
		{
			printf("Too many lines in file 1\n");
			printf("Press <Enter> to quit");
			getchar();
			return 3;
		}
	}

	circular=true;
		for (i=0; i<NumLine1; i++)
		{
			for (j=0; j<2; j++)
			{
				if(next[i][j][0] != -1) break;
				dmax=100000;
				for (k=0; k<NumLine1; k++)
				{
					if(k != i)
					{
						for(l=0; l<2; l++)
						{
							d=MANHATTAN(SortBuf[i][j], SortBuf[k][l]);
							if(d<dmax)
							{
								dmax=d;
								next[i][j][0]=k;
								next[i][j][1]=l;
							}
							if(d==0) break;
						}
						if(d==0) break;
					}
				}
				if(dmax>SMALL) 
				{
					next[i][j][0]=-1;
					circular=false;
				}
			}
		}
		if(circular) 
		{
			next[next[0][0][0]][next[0][0][1]][0]=-1;
			next[0][0][0]=-1;
		}
		InLineIdx=0;
		NumPath=0;
		for (i=0; i<NumLine1; i++)
		{
			for (j=0; j<2; j++)
			{		
				int a,b,c,d;
				if(next[i][j][0] == -1)
				{
					NumPath++;
					a=i; b=j;
					do
					{
						double p1[3], p2[3], q1[3], delta[3], temp[3];
						int nsplit;

						SET(p1, SortBuf[a][b]);
						SET(p2, SortBuf[a][1-b]);
		
						nsplit=(DIST(p1,p2)/maxlength)+1;

						SUB(temp,p2,p1);
						MULT(delta, temp, 1./nsplit);
						for (k=0; k<nsplit; k++)
						{
							MULT (q1, delta, k);
							ADD (InLine1[InLineIdx][0],p1,q1);
							ADD (InLine1[InLineIdx][1],InLine1[InLineIdx][0],delta);
							InLineIdx++;
						}
						d=next[a][1-b][1];
						c=next[a][1-b][0];
						next[a][1-b][0] = -2;
						next[a][b][0] = -2;
						b=d;
						a=c;
					}
					while((a!=-1));
				}
			}
		}

		NumLine1=InLineIdx;
	if(NumPath > 1)
		printf("%d distinct paths found in Line file 1. Unexpected results may happen!\n",NumPath);
	
	printf("%d Lines in Line file 1\n", NumLine1);
	SET(InLine1[NumLine1][0],InLine1[NumLine1-1][1]);



	printf("\nRead Line File 2...\n");
	NumLine2=0;
	while (fgets(buf, 1024, LdrawLineFile2))
	{
		type = -1;
		sscanf(buf, "%d", &type);
		if(type==2)
		{
			sscanf (buf, "%d %d %lf %lf %lf %lf %lf %lf", &type, &color, 
				&SortBuf[NumLine2][0][0], &SortBuf[NumLine2][0][1], &SortBuf[NumLine2][0][2], 
				&SortBuf[NumLine2][1][0], &SortBuf[NumLine2][1][1], &SortBuf[NumLine2][1][2]);
				next[NumLine2][0][0] = next[NumLine2][1][0] = -1;
			NumLine2++;
		}
		if(NumLine2>MAX_LINE)
		{
			printf("Too many lines in file 2\n");
			printf("Press <Enter> to quit");
			getchar();
			return 3;
		}

	}

	
	circular=true;
		for (i=0; i<NumLine2; i++)
		{
			for (j=0; j<2; j++)
			{
				if(next[i][j][0] != -1) break;
				dmax=100000;
				for (k=0; k<NumLine2; k++)
				{
					if(k != i)
					{
						for(l=0; l<2; l++)
						{
							d=MANHATTAN(SortBuf[i][j], SortBuf[k][l]);
							if(d<dmax)
							{
								dmax=d;
								next[i][j][0]=k;
								next[i][j][1]=l;
							}
							if(d==0) break;
						}
						if(d==0) break;
					}
				}
				if(dmax>SMALL) 
				{
					next[i][j][0]=-1;
					circular=false;
				}
			}
		}
		if(circular) 
		{
			next[next[0][0][0]][next[0][0][1]][0]=-1;
			next[0][0][0]=-1;
		}
		InLineIdx=0;
		NumPath=0;
		for (i=0; i<NumLine2; i++)
		{
			for (j=0; j<2; j++)
			{		
				int a,b,c,d;
				if(next[i][j][0] == -1)
				{
					NumPath++;
					a=i; b=j;
					do
					{
						double p1[3], p2[3], q1[3], delta[3], temp[3];
						int nsplit;

						SET(p1, SortBuf[a][b]);
						SET(p2, SortBuf[a][1-b]);
		
						nsplit=(DIST(p1,p2)/maxlength)+1;

						SUB(temp,p2,p1);
						MULT(delta, temp, 1./nsplit);
						for (k=0; k<nsplit; k++)
						{
							MULT (q1, delta, k);
							ADD (InLine2[InLineIdx][0],p1,q1);
							ADD (InLine2[InLineIdx][1],InLine2[InLineIdx][0],delta);
							InLineIdx++;
						}
						d=next[a][1-b][1];
						c=next[a][1-b][0];
						next[a][1-b][0] = -2;
						next[a][b][0] = -2;
						b=d;
						a=c;
					}
					while((a!=-1));
				}
			}
		}
		NumLine2=InLineIdx;
	if(NumPath > 1)
		printf("%d distinct paths found in Line file 2. Unexpected results may happen!\n",NumPath);
	

	if(direction)
	{
		for(i=0; i<NumLine2/2; i++)
		{
			double Temp1[3], Temp2[3];
			SET(Temp1, InLine2[NumLine2-i-1][0]); 
			SET(Temp2, InLine2[NumLine2-i-1][1]); 
			SET(InLine2[NumLine2-i-1][0], InLine2[i][1]); 
			SET(InLine2[NumLine2-i-1][1], InLine2[i][0]); 
			SET(InLine2[i][0], Temp2); 
			SET(InLine2[i][1], Temp1); 
		}

		if((NumLine2 % 2)==1)
		{
			double Temp1[3];
			SET(Temp1, InLine2[NumLine2/2][0]);
			SET(InLine2[NumLine2/2][0], InLine2[NumLine2/2][1]);
			SET(InLine2[NumLine2/2][1], Temp1);
		}
			
	}

	printf("%d Lines in Line file 2\n", NumLine2);

	SET(InLine2[NumLine2][0],InLine2[NumLine2-1][1]);

	{
		float a,b;
		printf("\nCreating Triangles\n");
		i=j=0;
		NumTri=0; Degen=0;
		while((i<NumLine1) || (j<NumLine2))
		{
			SET(Surf[NumTri][0],InLine1[i][0]);
			SET(Surf[NumTri][1],InLine2[j][0]);
			a=(float)(i+1)/(float)NumLine1;
			b=(float)(j+1)/(float)NumLine2;
			if(bias>0)
				a = (a + (bias+1) * pow(a, bias+1)) / ( 2 + bias ); 
			else
				b = (b + (1-bias) * pow(b, 1-bias)) / ( 2 - bias ); 
	
			// printf("\n%f %f", a, b); 
			if(line>2)
			{
				if(line==4)
				{
					i++;
					SET(Surf[NumTri][2],InLine1[i][0]);
					line=1;
				}
				if(line==3)
				{
					j++;
					SET(Surf[NumTri][2],InLine2[j][0]);
					line=2;
				}

			}

			else if (sequential)
			{
				if(a < b)
				{
					i++;
					SET(Surf[NumTri][2],InLine1[i][0]);
					line=1;
				}
				else
				{
					j++;
					SET(Surf[NumTri][2],InLine2[j][0]);
					line=2;
				}
			}
			else if (!sequential)
			{
				if(i==NumLine1)
				{
					j++;
					SET(Surf[NumTri][2],InLine2[j][0]);
					line=2;
				}
				else if(j==NumLine2)
				{
					i++;
					SET(Surf[NumTri][2],InLine1[i][0]);
					line=1;
				}
				else if(MANHATTAN(Surf[NumTri][1], InLine1[i+1][0]) < MANHATTAN(Surf[NumTri][0], InLine2[j+1][0]))
				{
					i++;
					SET(Surf[NumTri][2],InLine1[i][0]);
					line=1;
				}
				else
				{
					j++;
					SET(Surf[NumTri][2],InLine2[j][0]);
					line=2;
				}
			}
			// printf("%d %d\n", i, j);
			if ((MANHATTAN(Surf[NumTri][0], Surf[NumTri][1])<SMALL) || (MANHATTAN(Surf[NumTri][1], Surf[NumTri][2])<SMALL) || (MANHATTAN(Surf[NumTri][0], Surf[NumTri][2])<SMALL) )
			{
				TriQuad[NumTri]=0;
				Degen++;
				line+=2;
				NumTri--;
			}
			else
			{
				TriQuad[NumTri]=3;
			}
			NumTri++;
		}
	}
	printf("%d Triangle(s) created\n", NumTri-Degen);
	printf("(%d degenerated suppressed)\n", Degen);

	printf("\nCreating Conditional Lines\n");
	printf("...and condensing coplanar Triangles into Quads\n");
	
	NumCond=0;
	NumQuad=0;
	CondFlag[0]=0;

	SET(Surf[NumTri][0], Surf[0][0]);
	SET(Surf[NumTri][1], Surf[0][1]);
	SET(Surf[NumTri][2], Surf[0][2]);
	TriQuad[NumTri]=3;



	for(i=1; i<=NumTri; i++)
	{

		CondFlag[i]=0;
		if( ((TriQuad[i-1]==3) || (TriQuad[i-1]==6)) && (TriQuad[i]==3))
		{

// Find common points and not common			
			int c[2], n[2];
			int ci=0;

			for(j=0; j<3; j++)
			{
				int flag=true;
				for(k=0; k<3; k++)
				{
					if (MANHATTAN(Surf[i-1][j],Surf[i][k]) < SMALL) 
					{
						c[ci]=j;
						flag=false;
						ci++;
						break;
					}
				}
				if(flag) n[0]=j;
			}
			if(ci<2) continue;

			for(k=0; k<3; k++)
			{
				int flag=true;
				for(j=0; j<3; j++)
				{
					if(MANHATTAN(Surf[i-1][j],Surf[i][k]) < SMALL) 
					{
						flag=false;
						break;
					}
				}
				if(flag) n[1]=k;
			}

			double a;
			a = Tri_Angle (Surf[i-1][0],Surf[i-1][1],Surf[i-1][2], Surf[i][0],Surf[i][1],Surf[i][2]);
//			printf("angle %lf\n", a);
			if(a > SMALLANGLE)
			{
				
				if(MANHATTAN(Surf[i-1][c[0]],Surf[i-1][c[1]]) > SMALL)
				{
					CondFlag[i]=1;

					SET(CondLine[i][0], Surf[i-1][c[0]]);
					SET(CondLine[i][1], Surf[i-1][c[1]]);
					SET(CondLine[i][2], Surf[i-1][n[0]]);
					SET(CondLine[i][3], Surf[i][n[1]]);	
					NumCond++;
				}
			}
			else if ((TriQuad[i-1]!=6) && (i<NumTri))
			{
				if(Convex(Surf[i-1][c[0]],Surf[i-1][n[0]],Surf[i-1][c[1]],Surf[i][n[1]]))
				{
					double temp[3][3];
					double a[3], b[3], v1[3], v2[3];
					for (j=0; j<3; j++)
					{
						SET (temp[j], Surf[i-1][j]);
					}

					// Maintain quad BFC direction identical to 1st triangle
					SUB(a, Surf[i-1][0], Surf[i-1][1]); 
					SUB(b, Surf[i-1][0], Surf[i-1][2]);
					CROSS(v1,a,b);
					SUB(a, temp[c[0]], temp[n[0]]); 
					SUB(b, temp[c[0]], temp[c[1]]);
					CROSS(v2,a,b);
					if(DOT(v1,v2)>0)
					{
						SET(Surf[i-1][0], temp[c[0]]);
						SET(Surf[i-1][1], temp[n[0]]);
						SET(Surf[i-1][2], temp[c[1]]);
					}
					else
					{
						SET(Surf[i-1][2], temp[c[0]]);
						SET(Surf[i-1][1], temp[n[0]]);
						SET(Surf[i-1][0], temp[c[1]]);
					}

					SET(Surf[i-1][3], Surf[i][n[1]]);
					TriQuad[i-1]=4;
					TriQuad[i]=6;
					CondFlag[i]=0;
					NumQuad++;
				}
			}
		}
		else CondFlag[i]=0;
	}
	printf("%d Conditional Line(s) created\n", NumCond);
	printf("%d Triangle(s) condensed into %d Quad(s)\n", NumQuad*2, NumQuad);

	printf("\nWriting output file\n");

	fprintf(LdrawSurfaceFileOut, "0 Surface %s created upon %s and %s\n", argv[argc-1], argv[argc-3], argv[argc-2]);
	fprintf(LdrawSurfaceFileOut, "0 Name: %s\n", argv[argc-1]);
	fprintf(LdrawSurfaceFileOut, "0 Author: Philo's Coverer\n0\n");
//	fprintf(LdrawSurfaceFileOut, "0 BFC CERTIFY CCW\n\n");
	TriWritten=0;
	NumQuad=0;
	NumCond=0;

	for(k=0; k<NumTri; k++)
	{
		if(TriQuad[k]==3)
		{
			fprintf(LdrawSurfaceFileOut, "3 16 %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg\n", Surf[k][0][0],Surf[k][0][1],Surf[k][0][2],Surf[k][1][0],Surf[k][1][1],Surf[k][1][2],Surf[k][2][0],Surf[k][2][1],Surf[k][2][2]);
			TriWritten++;
		}
		if(TriQuad[k]==4)
		{
			fprintf(LdrawSurfaceFileOut, "4 16 %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg\n", Surf[k][0][0],Surf[k][0][1],Surf[k][0][2],Surf[k][1][0],Surf[k][1][1],Surf[k][1][2],Surf[k][2][0],Surf[k][2][1],Surf[k][2][2], Surf[k][3][0],Surf[k][3][1],Surf[k][3][2]);
			NumQuad++;
		}
	}

	for(k=1; k<=NumTri; k++)
	{
		if(CondFlag[k] != 0)
		{
			fprintf(LdrawSurfaceFileOut, "5 24 %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg %.7lg\n", CondLine[k][0][0],CondLine[k][0][1],CondLine[k][0][2],CondLine[k][1][0],CondLine[k][1][1],CondLine[k][1][2],CondLine[k][2][0],CondLine[k][2][1],CondLine[k][2][2],CondLine[k][3][0],CondLine[k][3][1],CondLine[k][3][2]);
			NumCond++;
		}

	}

	printf("%d Triangle(s) written\n", TriWritten);
	printf("%d Quad(s) written\n", NumQuad);
	printf("%d Conditional Line(s) written\n", NumCond);
	fprintf(LdrawSurfaceFileOut, "0\n");
	fclose (LdrawLineFile1);
	fclose (LdrawLineFile2);
	fclose (LdrawSurfaceFileOut);
	printf("Press <Enter> to quit");
	getchar();
	return(0);
}
